<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Quiz;
use App\Models\QuizAttempt;
use App\Services\QuizService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class QuizAttemptController extends Controller
{
    public function __construct(
        private QuizService $quizService
    ) {}

    /**
     * Start a new quiz attempt.
     */
    public function start(Request $request, Quiz $quiz): JsonResponse
    {
        try {
            $attempt = $this->quizService->startAttempt($quiz, Auth::user(), $request->all());

            return response()->json([
                'success' => true,
                'message' => 'Quiz attempt started successfully',
                'data' => $attempt->load(['quiz', 'user']),
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to start quiz attempt',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Resume an existing quiz attempt.
     */
    public function resume(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $attempt = $this->quizService->resumeAttempt($attempt);

            return response()->json([
                'success' => true,
                'message' => 'Quiz attempt resumed successfully',
                'data' => $attempt->load(['quiz', 'user', 'answers.question']),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to resume quiz attempt',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Get current attempt for a quiz.
     */
    public function current(Quiz $quiz): JsonResponse
    {
        try {
            $attempt = $quiz->getInProgressAttempt(Auth::user());

            if (!$attempt) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active attempt found',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $attempt->load(['quiz', 'user', 'answers.question']),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get current attempt',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Answer a question.
     */
    public function answer(Request $request, QuizAttempt $attempt): JsonResponse
    {
        try {
            $request->validate([
                'question_id' => 'required|integer|exists:quiz_questions,id',
                'answer' => 'required',
                'metadata' => 'nullable|array',
            ]);

            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $answerRecord = $this->quizService->answerQuestion(
                $attempt,
                $request->question_id,
                $request->answer,
                $request->get('metadata', [])
            );

            return response()->json([
                'success' => true,
                'message' => 'Question answered successfully',
                'data' => $answerRecord->load('question'),
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to answer question',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Skip a question.
     */
    public function skip(Request $request, QuizAttempt $attempt): JsonResponse
    {
        try {
            $request->validate([
                'question_id' => 'required|integer|exists:quiz_questions,id',
            ]);

            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $this->quizService->skipQuestion($attempt, $request->question_id);

            return response()->json([
                'success' => true,
                'message' => 'Question skipped successfully',
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to skip question',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Navigate to a specific question.
     */
    public function navigate(Request $request, QuizAttempt $attempt): JsonResponse
    {
        try {
            $request->validate([
                'question_id' => 'required|integer|exists:quiz_questions,id',
            ]);

            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $attempt->navigateToQuestion($request->question_id);

            return response()->json([
                'success' => true,
                'message' => 'Navigated to question successfully',
                'data' => $attempt->fresh(['quiz', 'user', 'answers.question']),
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to navigate to question',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Get next question.
     */
    public function nextQuestion(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $nextQuestion = $attempt->getNextQuestion();

            if (!$nextQuestion) {
                return response()->json([
                    'success' => false,
                    'message' => 'No more questions available',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $nextQuestion,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get next question',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get previous question.
     */
    public function previousQuestion(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $previousQuestion = $attempt->getPreviousQuestion();

            if (!$previousQuestion) {
                return response()->json([
                    'success' => false,
                    'message' => 'No previous question available',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $previousQuestion,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get previous question',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Complete the quiz attempt.
     */
    public function complete(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $attempt = $this->quizService->completeAttempt($attempt);

            return response()->json([
                'success' => true,
                'message' => 'Quiz attempt completed successfully',
                'data' => $attempt->load(['quiz', 'user', 'answers.question']),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to complete quiz attempt',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Abandon the quiz attempt.
     */
    public function abandon(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $attempt = $this->quizService->abandonAttempt($attempt);

            return response()->json([
                'success' => true,
                'message' => 'Quiz attempt abandoned successfully',
                'data' => $attempt->load(['quiz', 'user']),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to abandon quiz attempt',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Get attempt details.
     */
    public function show(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            $attempt->load(['quiz', 'user', 'answers.question']);

            return response()->json([
                'success' => true,
                'data' => $attempt,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get attempt details',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get user's attempts for a quiz.
     */
    public function userAttempts(Request $request, Quiz $quiz = null): JsonResponse
    {
        try {
            $attempts = $this->quizService->getUserAttempts(Auth::user(), $quiz?->id);

            return response()->json([
                'success' => true,
                'data' => $attempts,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get user attempts',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get attempt results.
     */
    public function results(QuizAttempt $attempt): JsonResponse
    {
        try {
            // Check if user owns this attempt
            if ($attempt->user_id !== Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to attempt',
                ], 403);
            }

            // Only show results for completed attempts
            if ($attempt->status !== 'completed') {
                return response()->json([
                    'success' => false,
                    'message' => 'Attempt is not completed yet',
                ], 400);
            }

            $attempt->load(['quiz', 'answers.question']);

            $results = [
                'attempt' => $attempt,
                'summary' => [
                    'total_questions' => $attempt->total_questions,
                    'questions_answered' => $attempt->questions_answered,
                    'total_score' => $attempt->total_score,
                    'max_score' => $attempt->max_score,
                    'percentage' => $attempt->percentage,
                    'time_spent' => $attempt->time_spent,
                    'completed_at' => $attempt->completed_at,
                ],
                'answers' => $attempt->answers->map(function ($answer) {
                    return [
                        'question' => $answer->question,
                        'answer' => $answer->getFormattedAnswer(),
                        'score' => $answer->score,
                        'max_score' => $answer->max_score,
                        'is_correct' => $answer->is_correct,
                        'is_partial' => $answer->is_partial,
                        'feedback' => $answer->feedback,
                        'manual_feedback' => $answer->manual_feedback,
                    ];
                }),
            ];

            return response()->json([
                'success' => true,
                'data' => $results,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get attempt results',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}