<?php

namespace App\Livewire\Admin;

use App\Services\FlashMessageService;
use Livewire\Component;

class HelpCenter extends Component
{
    public $search = '';
    public $selectedCategory = 'all';
    public $contactForm = [
        'name' => '',
        'email' => '',
        'subject' => '',
        'message' => ''
    ];

    public $faqs = [
        [
            'category' => 'general',
            'question' => 'How do I create a new user?',
            'answer' => 'To create a new user, go to the "Manage Users" section in the sidebar, click the "Add New User" button, fill in the required information, and click "Create User".'
        ],
        [
            'category' => 'general',
            'question' => 'How do I assign roles to users?',
            'answer' => 'When creating or editing a user, you can select a role from the dropdown menu. Roles determine what permissions a user has in the system.'
        ],
        [
            'category' => 'permissions',
            'question' => 'How do I manage permissions?',
            'answer' => 'As an Owner, you can manage permissions by going to "Owner Settings" > "Manage Permissions". Here you can assign or revoke permissions for different roles.'
        ],
        [
            'category' => 'permissions',
            'question' => 'What is the difference between Owner and Administrator?',
            'answer' => 'Owner has all permissions in the system including permission management. Administrator has most permissions but cannot manage permissions for other roles.'
        ],
        [
            'category' => 'system',
            'question' => 'How do I change my password?',
            'answer' => 'Click on your profile in the top right corner, select "Profile", and then use the "Change Password" section to update your password.'
        ],
        [
            'category' => 'system',
            'question' => 'How do I update my profile information?',
            'answer' => 'Go to your profile page and update the information in the "Profile Information" section. You can also upload a new avatar image.'
        ]
    ];

    public function getFilteredFaqs()
    {
        $filtered = $this->faqs;

        if ($this->selectedCategory !== 'all') {
            $filtered = array_filter($filtered, function($faq) {
                return $faq['category'] === $this->selectedCategory;
            });
        }

        if (!empty($this->search)) {
            $filtered = array_filter($filtered, function($faq) {
                return stripos($faq['question'], $this->search) !== false || 
                       stripos($faq['answer'], $this->search) !== false;
            });
        }

        return $filtered;
    }

    public function submitContactForm()
    {
        $this->validate([
            'contactForm.name' => 'required|string|max:255',
            'contactForm.email' => 'required|email',
            'contactForm.subject' => 'required|string|max:255',
            'contactForm.message' => 'required|string|max:1000',
        ]);

        try {
            // In a real application, you would send an email or save to database
            // For now, we'll just show a success message
            
            FlashMessageService::success('Your message has been sent successfully!', 'Message Sent');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'success', 'Your message has been sent successfully!', 'Message Sent', 5000);
            
            $this->reset('contactForm');
            
        } catch (\Exception $e) {
            FlashMessageService::error('Failed to send message. Please try again.', 'Send Failed');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'error', 'Failed to send message. Please try again.', 'Send Failed', 8000);
            
            \Log::error('Contact form error: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.help-center', [
            'filteredFaqs' => $this->getFilteredFaqs()
        ]);
    }
}
