<?php

namespace App\Livewire\Admin;

use App\Services\FlashMessageService;
use Livewire\Component;
use Illuminate\Support\Facades\Hash;

class Profile extends Component
{

    public $name;
    public $email;
    public $phone;
    public $address;
    public $date_of_birth;
    public $current_password;
    public $new_password;
    public $new_password_confirmation;

    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email',
        'phone' => 'nullable|string|max:20',
        'address' => 'nullable|string|max:500',
        'date_of_birth' => 'nullable|date',
        'current_password' => 'nullable|string',
        'new_password' => 'nullable|min:8|confirmed',
    ];

    protected $messages = [
        'name.required' => 'Name is required.',
        'email.required' => 'Email is required.',
        'email.email' => 'Please enter a valid email address.',
        'email.unique' => 'This email is already taken.',
        'new_password.min' => 'Password must be at least 8 characters.',
        'new_password.confirmed' => 'Password confirmation does not match.',
    ];

    public function mount()
    {
        $user = auth()->user();
        $this->name = $user->name;
        $this->email = $user->email;
        $this->phone = $user->phone;
        $this->address = $user->address;
        $this->date_of_birth = $user->date_of_birth?->format('Y-m-d');
    }

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function updateProfile()
    {
        $this->rules['email'] = 'required|email|unique:users,email,' . auth()->id();
        
        $this->validate();

        try {
            $user = auth()->user();
            

            $user->update([
                'name' => $this->name,
                'email' => $this->email,
                'phone' => $this->phone ?: null,
                'address' => $this->address ?: null,
                'date_of_birth' => $this->date_of_birth ?: null,
            ]);

            FlashMessageService::success('Profile updated successfully!', 'Profile Updated');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'success', 'Profile updated successfully!', 'Profile Updated', 5000);
            
        } catch (\Exception $e) {
            FlashMessageService::error('Failed to update profile. Please try again.', 'Update Failed');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'error', 'Failed to update profile. Please try again.', 'Update Failed', 8000);
            
            \Log::error('Profile update error: ' . $e->getMessage());
        }
    }

    public function updatePassword()
    {
        $this->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|min:8|confirmed',
        ]);

        try {
            $user = auth()->user();
            
            if (!Hash::check($this->current_password, $user->password)) {
                FlashMessageService::error('Current password is incorrect.', 'Password Error');
                
                // Dispatch flash message event
                $this->dispatch('flashMessage', 'error', 'Current password is incorrect.', 'Password Error', 8000);
                
                return;
            }

            $user->update([
                'password' => Hash::make($this->new_password)
            ]);

            FlashMessageService::success('Password updated successfully!', 'Password Updated');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'success', 'Password updated successfully!', 'Password Updated', 5000);
            
            $this->reset(['current_password', 'new_password', 'new_password_confirmation']);
            
        } catch (\Exception $e) {
            FlashMessageService::error('Failed to update password. Please try again.', 'Password Update Failed');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'error', 'Failed to update password. Please try again.', 'Password Update Failed', 8000);
            
            \Log::error('Password update error: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.profile');
    }
}
