<?php

namespace App\Livewire\Admin;

use App\Models\Quiz;
use App\Models\QuizInvitation;
use App\Services\QuizInvitationService;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\On;

class QuizInvitationManagement extends Component
{
    use WithPagination;

    public $quiz;
    public $search = '';
    public $statusFilter = '';
    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    
    // Invitation form
    public $showInviteForm = false;
    public $emails = '';
    public $customMessage = '';
    public $selectedInvitations = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'sortField' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc'],
    ];

    public function mount(Quiz $quiz)
    {
        $this->quiz = $quiz;
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function clearFilters()
    {
        $this->search = '';
        $this->statusFilter = '';
        $this->sortField = 'created_at';
        $this->sortDirection = 'desc';
    }

    public function toggleInviteForm()
    {
        $this->showInviteForm = !$this->showInviteForm;
        if (!$this->showInviteForm) {
            $this->resetInviteForm();
        }
    }

    public function resetInviteForm()
    {
        $this->emails = '';
        $this->customMessage = '';
    }

    public function sendInvitations()
    {
        $this->validate([
            'emails' => 'required|string',
            'customMessage' => 'nullable|string|max:1000',
        ]);

        $emailList = array_filter(array_map('trim', explode(',', $this->emails)));
        
        if (empty($emailList)) {
            session()->flash('error', 'Please enter at least one email address.');
            return;
        }

        $service = new QuizInvitationService();
        $results = $service->sendInvitations($this->quiz, $emailList, $this->customMessage);

        $message = "Invitations sent: " . count($results['success']);
        
        if (!empty($results['duplicates'])) {
            $message .= " | Duplicates skipped: " . count($results['duplicates']);
        }
        
        if (!empty($results['errors'])) {
            $message .= " | Errors: " . count($results['errors']);
        }

        session()->flash('success', $message);
        
        $this->resetInviteForm();
        $this->showInviteForm = false;
    }

    public function resendInvitation($invitationId)
    {
        $invitation = QuizInvitation::findOrFail($invitationId);
        $service = new QuizInvitationService();
        
        if ($service->resendInvitation($invitation)) {
            session()->flash('success', 'Invitation resent successfully.');
        } else {
            session()->flash('error', 'Failed to resend invitation.');
        }
    }

    public function deleteInvitation($invitationId)
    {
        $invitation = QuizInvitation::findOrFail($invitationId);
        $service = new QuizInvitationService();
        
        if ($service->deleteInvitation($invitation)) {
            session()->flash('success', 'Invitation deleted successfully.');
        } else {
            session()->flash('error', 'Failed to delete invitation.');
        }
    }

    public function bulkDelete()
    {
        if (empty($this->selectedInvitations)) {
            session()->flash('error', 'Please select invitations to delete.');
            return;
        }

        $service = new QuizInvitationService();
        $results = $service->bulkDeleteInvitations($this->selectedInvitations);

        $message = "Deleted: " . $results['success'];
        if (!empty($results['errors'])) {
            $message .= " | Errors: " . count($results['errors']);
        }

        session()->flash('success', $message);
        $this->selectedInvitations = [];
    }

    public function toggleSelectAll()
    {
        if (count($this->selectedInvitations) === $this->getInvitations()->count()) {
            $this->selectedInvitations = [];
        } else {
            $this->selectedInvitations = $this->getInvitations()->pluck('id')->toArray();
        }
    }

    public function getInvitations()
    {
        return $this->quiz->invitations()
            ->when($this->search, function ($query) {
                $query->where('email', 'like', '%' . $this->search . '%');
            })
            ->when($this->statusFilter, function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate(10);
    }

    public function getInvitationStats()
    {
        $service = new QuizInvitationService();
        return $service->getInvitationStats($this->quiz);
    }

    public function render()
    {
        return view('livewire.admin.quiz-invitation-management', [
            'invitations' => $this->getInvitations(),
            'stats' => $this->getInvitationStats(),
        ]);
    }
}
