<?php

namespace App\Livewire\Admin;

use App\Models\Quiz;
use App\Services\QuizService;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;

class QuizManagement extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = '';
    public $visibilityFilter = '';
    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    public $perPage = 10;

    protected $listeners = ['quizDeleted' => 'render'];

    public function mount()
    {
        $this->perPage = session('quiz_per_page', 10);
    }

    public function updatedPerPage($value)
    {
        session(['quiz_per_page' => $value]);
        $this->perPage = $value;
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        
        $this->sortField = $field;
    }

    public function clearFilters()
    {
        $this->search = '';
        $this->statusFilter = '';
        $this->visibilityFilter = '';
        $this->sortField = 'created_at';
        $this->sortDirection = 'desc';
    }

    public function deleteQuiz($quizId)
    {
        try {
            $quiz = Quiz::findOrFail($quizId);
            
            // Check permissions
            if (!Auth::user()->can('quizzes.delete') && $quiz->created_by !== Auth::id()) {
                session()->flash('error', 'You do not have permission to delete this quiz.');
                return;
            }

            $quizService = app(QuizService::class);
            $quizService->deleteQuiz($quiz, Auth::id());

            session()->flash('success', 'Quiz deleted successfully.');
            $this->dispatch('quizDeleted');
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to delete quiz: ' . $e->getMessage());
        }
    }

    public function publishQuiz($quizId)
    {
        try {
            $quiz = Quiz::findOrFail($quizId);
            
            // Check permissions
            if (!Auth::user()->can('quizzes.publish') && $quiz->created_by !== Auth::id()) {
                session()->flash('error', 'You do not have permission to publish this quiz.');
                return;
            }

            $quizService = app(QuizService::class);
            $quizService->publishQuiz($quiz, Auth::id());

            session()->flash('success', 'Quiz published successfully.');
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to publish quiz: ' . $e->getMessage());
        }
    }

    public function unpublishQuiz($quizId)
    {
        try {
            $quiz = Quiz::findOrFail($quizId);
            
            // Check permissions
            if (!Auth::user()->can('quizzes.publish') && $quiz->created_by !== Auth::id()) {
                session()->flash('error', 'You do not have permission to unpublish this quiz.');
                return;
            }

            $quizService = app(QuizService::class);
            $quizService->unpublishQuiz($quiz, Auth::id());

            session()->flash('success', 'Quiz unpublished successfully.');
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to unpublish quiz: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $query = Quiz::query();

        // Apply search
        if (!empty($this->search)) {
            $query->where(function ($q) {
                $q->where('title', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%');
            });
        }

        // Apply filters
        if (!empty($this->statusFilter)) {
            $query->where('status', $this->statusFilter);
        }

        if (!empty($this->visibilityFilter)) {
            $query->where('visibility', $this->visibilityFilter);
        }

        // Apply sorting
        $query->orderBy($this->sortField, $this->sortDirection);

        $quizzes = $query->with(['creator', 'questions'])
                        ->paginate($this->perPage);

        return view('livewire.admin.quiz-management', [
            'quizzes' => $quizzes,
        ]);
    }
}