<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;

class QuizAnalytic extends Model
{
    use HasFactory;

    const EVENT_TYPES = [
        'quiz_viewed' => 'Quiz Viewed',
        'quiz_started' => 'Quiz Started',
        'question_viewed' => 'Question Viewed',
        'question_answered' => 'Question Answered',
        'question_skipped' => 'Question Skipped',
        'attempt_resumed' => 'Attempt Resumed',
        'attempt_completed' => 'Attempt Completed',
        'attempt_abandoned' => 'Attempt Abandoned',
        'attempt_expired' => 'Attempt Expired',
        'quiz_shared' => 'Quiz Shared',
        'feedback_submitted' => 'Feedback Submitted',
    ];

    protected $fillable = [
        'quiz_id',
        'question_id',
        'user_id',
        'event_type',
        'event_data',
        'metadata',
        'session_id',
        'ip_address',
        'user_agent',
        'occurred_at',
    ];

    protected $casts = [
        'event_data' => 'array',
        'metadata' => 'array',
        'occurred_at' => 'datetime',
    ];

    // Relationships
    public function quiz(): BelongsTo
    {
        return $this->belongsTo(Quiz::class);
    }

    public function question(): BelongsTo
    {
        return $this->belongsTo(QuizQuestion::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Scopes
    public function scopeByEventType(Builder $query, string $eventType): Builder
    {
        return $query->where('event_type', $eventType);
    }

    public function scopeByQuiz(Builder $query, int $quizId): Builder
    {
        return $query->where('quiz_id', $quizId);
    }

    public function scopeByUser(Builder $query, int $userId): Builder
    {
        return $query->where('user_id', $userId);
    }

    public function scopeBySession(Builder $query, string $sessionId): Builder
    {
        return $query->where('session_id', $sessionId);
    }

    public function scopeInDateRange(Builder $query, Carbon $startDate, Carbon $endDate): Builder
    {
        return $query->whereBetween('occurred_at', [$startDate, $endDate]);
    }

    public function scopeLastDays(Builder $query, int $days): Builder
    {
        return $query->where('occurred_at', '>=', now()->subDays($days));
    }

    // Static methods for tracking events
    public static function trackQuizViewed(int $quizId, ?int $userId = null, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_viewed',
            'event_data' => [
                'quiz_id' => $quizId,
                'viewed_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizStarted(int $quizId, int $userId, int $attemptId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_started',
            'event_data' => [
                'quiz_id' => $quizId,
                'attempt_id' => $attemptId,
                'started_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuestionViewed(int $quizId, int $questionId, int $userId, int $attemptId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'question_id' => $questionId,
            'user_id' => $userId,
            'event_type' => 'question_viewed',
            'event_data' => [
                'quiz_id' => $quizId,
                'question_id' => $questionId,
                'attempt_id' => $attemptId,
                'viewed_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuestionAnswered(int $quizId, int $questionId, int $userId, int $attemptId, $answer, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'question_id' => $questionId,
            'user_id' => $userId,
            'event_type' => 'question_answered',
            'event_data' => [
                'quiz_id' => $quizId,
                'question_id' => $questionId,
                'attempt_id' => $attemptId,
                'answer' => $answer,
                'answered_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuestionSkipped(int $quizId, int $questionId, int $userId, int $attemptId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'question_id' => $questionId,
            'user_id' => $userId,
            'event_type' => 'question_skipped',
            'event_data' => [
                'quiz_id' => $quizId,
                'question_id' => $questionId,
                'attempt_id' => $attemptId,
                'skipped_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackAttemptResumed(int $quizId, int $userId, int $attemptId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'attempt_resumed',
            'event_data' => [
                'quiz_id' => $quizId,
                'attempt_id' => $attemptId,
                'resumed_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackAttemptCompleted(int $quizId, int $userId, int $attemptId, array $completionData = [], array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'attempt_completed',
            'event_data' => array_merge([
                'quiz_id' => $quizId,
                'attempt_id' => $attemptId,
                'completed_at' => now()->toISOString(),
            ], $completionData),
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackAttemptAbandoned(int $quizId, int $userId, int $attemptId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'attempt_abandoned',
            'event_data' => [
                'quiz_id' => $quizId,
                'attempt_id' => $attemptId,
                'abandoned_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackAttemptExpired(int $quizId, int $userId, int $attemptId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'attempt_expired',
            'event_data' => [
                'quiz_id' => $quizId,
                'attempt_id' => $attemptId,
                'expired_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizShared(int $quizId, int $userId, string $platform, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_shared',
            'event_data' => [
                'quiz_id' => $quizId,
                'platform' => $platform,
                'shared_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackFeedbackSubmitted(int $quizId, int $userId, int $attemptId, array $feedback, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'feedback_submitted',
            'event_data' => [
                'quiz_id' => $quizId,
                'attempt_id' => $attemptId,
                'feedback' => $feedback,
                'submitted_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizCreated(int $quizId, int $userId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_created',
            'event_data' => [
                'quiz_id' => $quizId,
                'created_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizUpdated(int $quizId, int $userId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_updated',
            'event_data' => [
                'quiz_id' => $quizId,
                'updated_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizPublished(int $quizId, int $userId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_published',
            'event_data' => [
                'quiz_id' => $quizId,
                'published_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizUnpublished(int $quizId, int $userId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_unpublished',
            'event_data' => [
                'quiz_id' => $quizId,
                'unpublished_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    public static function trackQuizDeleted(int $quizId, int $userId, array $metadata = []): self
    {
        return static::create([
            'quiz_id' => $quizId,
            'user_id' => $userId,
            'event_type' => 'quiz_deleted',
            'event_data' => [
                'quiz_id' => $quizId,
                'deleted_at' => now()->toISOString(),
            ],
            'metadata' => $metadata,
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'occurred_at' => now(),
        ]);
    }

    // Analytics methods
    public static function getQuizStats(int $quizId, Carbon $startDate = null, Carbon $endDate = null): array
    {
        $query = static::where('quiz_id', $quizId);
        
        if ($startDate && $endDate) {
            $query->whereBetween('occurred_at', [$startDate, $endDate]);
        }

        $stats = [
            'total_views' => $query->where('event_type', 'quiz_viewed')->count(),
            'total_starts' => $query->where('event_type', 'quiz_started')->count(),
            'total_completions' => $query->where('event_type', 'attempt_completed')->count(),
            'total_abandonments' => $query->where('event_type', 'attempt_abandoned')->count(),
            'completion_rate' => 0,
            'abandonment_rate' => 0,
        ];

        if ($stats['total_starts'] > 0) {
            $stats['completion_rate'] = round(($stats['total_completions'] / $stats['total_starts']) * 100, 2);
            $stats['abandonment_rate'] = round(($stats['total_abandonments'] / $stats['total_starts']) * 100, 2);
        }

        return $stats;
    }

    public static function getQuestionStats(int $questionId, Carbon $startDate = null, Carbon $endDate = null): array
    {
        $query = static::where('question_id', $questionId);
        
        if ($startDate && $endDate) {
            $query->whereBetween('occurred_at', [$startDate, $endDate]);
        }

        return [
            'total_views' => $query->where('event_type', 'question_viewed')->count(),
            'total_answers' => $query->where('event_type', 'question_answered')->count(),
            'total_skips' => $query->where('event_type', 'question_skipped')->count(),
        ];
    }

    public static function getUserActivity(int $userId, Carbon $startDate = null, Carbon $endDate = null): array
    {
        $query = static::where('user_id', $userId);
        
        if ($startDate && $endDate) {
            $query->whereBetween('occurred_at', [$startDate, $endDate]);
        }

        return $query->orderBy('occurred_at', 'desc')->get()->toArray();
    }
}