<?php

namespace App\Services;

use App\Models\Quiz;
use App\Models\QuizQuestion;
use App\Models\QuizAttempt;
use App\Models\QuizAnalytic;
use App\Models\QuizWebhook;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class QuizService
{
    public function createQuiz(array $data, int $userId): Quiz
    {
        $validator = Validator::make($data, [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'settings' => 'nullable|array',
            'metadata' => 'nullable|array',
            'i18n' => 'nullable|array',
            'visibility' => 'in:private,public,unlisted',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $quiz = Quiz::create(array_merge($data, [
            'created_by' => $userId,
            'updated_by' => $userId,
        ]));

        // Track analytics
        QuizAnalytic::trackQuizCreated($quiz->id, $userId);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('quiz.created', [
            'quiz_id' => $quiz->id,
            'title' => $quiz->title,
            'created_by' => $userId,
        ]);

        return $quiz;
    }

    public function updateQuiz(Quiz $quiz, array $data, int $userId): Quiz
    {
        $validator = Validator::make($data, [
            'title' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'settings' => 'nullable|array',
            'metadata' => 'nullable|array',
            'i18n' => 'nullable|array',
            'visibility' => 'in:private,public,unlisted',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $quiz->update(array_merge($data, [
            'updated_by' => $userId,
        ]));

        // Track analytics
        QuizAnalytic::trackQuizUpdated($quiz->id, $userId);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('quiz.updated', [
            'quiz_id' => $quiz->id,
            'title' => $quiz->title,
            'updated_by' => $userId,
        ]);

        return $quiz;
    }

    public function publishQuiz(Quiz $quiz, int $userId): Quiz
    {
        $quiz->update([
            'status' => 'published',
            'published_at' => now(),
            'updated_by' => $userId,
        ]);

        // Track analytics
        QuizAnalytic::trackQuizPublished($quiz->id, $userId);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('quiz.published', [
            'quiz_id' => $quiz->id,
            'title' => $quiz->title,
            'published_by' => $userId,
        ]);

        return $quiz;
    }

    public function unpublishQuiz(Quiz $quiz, int $userId): Quiz
    {
        $quiz->update([
            'status' => 'draft',
            'published_at' => null,
            'updated_by' => $userId,
        ]);

        // Track analytics
        QuizAnalytic::trackQuizUnpublished($quiz->id, $userId);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('quiz.unpublished', [
            'quiz_id' => $quiz->id,
            'title' => $quiz->title,
            'unpublished_by' => $userId,
        ]);

        return $quiz;
    }

    public function deleteQuiz(Quiz $quiz, int $userId): bool
    {
        // Track analytics
        QuizAnalytic::trackQuizDeleted($quiz->id, $userId);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('quiz.deleted', [
            'quiz_id' => $quiz->id,
            'title' => $quiz->title,
            'deleted_by' => $userId,
        ]);

        return $quiz->delete();
    }

    public function createQuestion(Quiz $quiz, array $data, int $userId): QuizQuestion
    {
        $validator = Validator::make($data, [
            'type' => 'required|string|in:' . implode(',', array_keys(QuizQuestion::QUESTION_TYPES)),
            'question_text' => 'required|string',
            'question_data' => 'nullable|array',
            'answer_data' => 'nullable|array',
            'scoring_data' => 'nullable|array',
            'validation_rules' => 'nullable|array',
            'points' => 'nullable|numeric|min:0',
            'is_required' => 'boolean',
            'order' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $question = QuizQuestion::create(array_merge($data, [
            'quiz_id' => $quiz->id,
            'created_by' => $userId,
            'updated_by' => $userId,
        ]));

        return $question;
    }

    public function updateQuestion(QuizQuestion $question, array $data, int $userId): QuizQuestion
    {
        $validator = Validator::make($data, [
            'type' => 'sometimes|string|in:' . implode(',', array_keys(QuizQuestion::QUESTION_TYPES)),
            'question_text' => 'sometimes|string',
            'question_data' => 'nullable|array',
            'answer_data' => 'nullable|array',
            'scoring_data' => 'nullable|array',
            'validation_rules' => 'nullable|array',
            'points' => 'nullable|numeric|min:0',
            'is_required' => 'boolean',
            'order' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $question->update(array_merge($data, [
            'updated_by' => $userId,
        ]));

        return $question;
    }

    public function deleteQuestion(QuizQuestion $question, int $userId): bool
    {
        return $question->delete();
    }

    public function reorderQuestions(Quiz $quiz, array $questionIds, int $userId): void
    {
        DB::transaction(function () use ($quiz, $questionIds, $userId) {
            foreach ($questionIds as $order => $questionId) {
                QuizQuestion::where('id', $questionId)
                    ->where('quiz_id', $quiz->id)
                    ->update([
                        'order' => $order + 1,
                        'updated_by' => $userId,
                    ]);
            }
        });
    }

    public function startAttempt(Quiz $quiz, User $user, array $settings = []): QuizAttempt
    {
        if (!$quiz->canBeAttemptedBy($user)) {
            throw new \Exception('User cannot attempt this quiz');
        }

        $attempt = QuizAttempt::createNew($quiz->id, $user->id, $settings);

        // Track analytics
        QuizAnalytic::trackQuizStarted($quiz->id, $user->id, $attempt->id);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('attempt.started', [
            'quiz_id' => $quiz->id,
            'attempt_id' => $attempt->id,
            'user_id' => $user->id,
        ]);

        return $attempt;
    }

    public function resumeAttempt(QuizAttempt $attempt): QuizAttempt
    {
        $attempt->resume();

        // Track analytics
        QuizAnalytic::trackAttemptResumed($attempt->quiz_id, $attempt->user_id, $attempt->id);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('attempt.resumed', [
            'quiz_id' => $attempt->quiz_id,
            'attempt_id' => $attempt->id,
            'user_id' => $attempt->user_id,
        ]);

        return $attempt;
    }

    public function answerQuestion(QuizAttempt $attempt, int $questionId, $answer, array $metadata = []): QuizAttemptAnswer
    {
        $answerRecord = $attempt->answerQuestion($questionId, $answer, $metadata);

        // Track analytics
        QuizAnalytic::trackQuestionAnswered(
            $attempt->quiz_id,
            $questionId,
            $attempt->user_id,
            $attempt->id,
            $answer
        );

        // Trigger webhooks
        QuizWebhook::triggerForEvent('question.answered', [
            'quiz_id' => $attempt->quiz_id,
            'question_id' => $questionId,
            'attempt_id' => $attempt->id,
            'user_id' => $attempt->user_id,
            'answer' => $answer,
        ]);

        return $answerRecord;
    }

    public function skipQuestion(QuizAttempt $attempt, int $questionId): void
    {
        $attempt->skipQuestion($questionId);

        // Track analytics
        QuizAnalytic::trackQuestionSkipped(
            $attempt->quiz_id,
            $questionId,
            $attempt->user_id,
            $attempt->id
        );

        // Trigger webhooks
        QuizWebhook::triggerForEvent('question.skipped', [
            'quiz_id' => $attempt->quiz_id,
            'question_id' => $questionId,
            'attempt_id' => $attempt->id,
            'user_id' => $attempt->user_id,
        ]);
    }

    public function completeAttempt(QuizAttempt $attempt): QuizAttempt
    {
        $attempt->complete();

        // Track analytics
        QuizAnalytic::trackAttemptCompleted($attempt->quiz_id, $attempt->user_id, $attempt->id, [
            'total_score' => $attempt->total_score,
            'max_score' => $attempt->max_score,
            'percentage' => $attempt->percentage,
            'time_spent' => $attempt->time_spent,
        ]);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('attempt.completed', [
            'quiz_id' => $attempt->quiz_id,
            'attempt_id' => $attempt->id,
            'user_id' => $attempt->user_id,
            'total_score' => $attempt->total_score,
            'max_score' => $attempt->max_score,
            'percentage' => $attempt->percentage,
        ]);

        return $attempt;
    }

    public function abandonAttempt(QuizAttempt $attempt): QuizAttempt
    {
        $attempt->abandon();

        // Track analytics
        QuizAnalytic::trackAttemptAbandoned($attempt->quiz_id, $attempt->user_id, $attempt->id);

        // Trigger webhooks
        QuizWebhook::triggerForEvent('attempt.abandoned', [
            'quiz_id' => $attempt->quiz_id,
            'attempt_id' => $attempt->id,
            'user_id' => $attempt->user_id,
        ]);

        return $attempt;
    }

    public function getQuizStats(Quiz $quiz, \Carbon\Carbon $startDate = null, \Carbon\Carbon $endDate = null): array
    {
        return QuizAnalytic::getQuizStats($quiz->id, $startDate, $endDate);
    }

    public function getQuestionStats(QuizQuestion $question, \Carbon\Carbon $startDate = null, \Carbon\Carbon $endDate = null): array
    {
        return QuizAnalytic::getQuestionStats($question->id, $startDate, $endDate);
    }

    public function getUserAttempts(User $user, int $quizId = null): \Illuminate\Database\Eloquent\Collection
    {
        $query = QuizAttempt::where('user_id', $user->id);

        if ($quizId) {
            $query->where('quiz_id', $quizId);
        }

        return $query->with(['quiz', 'answers.question'])
                    ->orderBy('started_at', 'desc')
                    ->get();
    }

    public function getQuizLeaderboard(Quiz $quiz, int $limit = 10): array
    {
        return QuizAttempt::where('quiz_id', $quiz->id)
            ->where('status', 'completed')
            ->with('user')
            ->orderBy('percentage', 'desc')
            ->orderBy('time_spent', 'asc')
            ->limit($limit)
            ->get()
            ->map(function ($attempt) {
                return [
                    'user' => $attempt->user->name,
                    'score' => $attempt->total_score,
                    'max_score' => $attempt->max_score,
                    'percentage' => $attempt->percentage,
                    'time_spent' => $attempt->time_spent,
                    'completed_at' => $attempt->completed_at,
                ];
            })
            ->toArray();
    }

    public function createVersion(Quiz $quiz, array $data = []): Quiz
    {
        $newVersion = $quiz->createVersion($data);

        // Copy questions to new version
        foreach ($quiz->allQuestions as $question) {
            $question->createVersion([
                'quiz_id' => $newVersion->id,
            ]);
        }

        return $newVersion;
    }

    public function getAvailableLanguages(): array
    {
        return [
            'en' => 'English',
            'es' => 'Spanish',
            'fr' => 'French',
            'de' => 'German',
            'it' => 'Italian',
            'pt' => 'Portuguese',
            'ru' => 'Russian',
            'ja' => 'Japanese',
            'ko' => 'Korean',
            'zh' => 'Chinese',
            'ar' => 'Arabic',
            'hi' => 'Hindi',
        ];
    }

    public function setQuizLanguage(Quiz $quiz, string $locale, array $translations): void
    {
        $quiz->setI18nData($locale, $translations);
    }

    public function getQuizInLanguage(Quiz $quiz, string $locale): array
    {
        $translations = $quiz->getI18nData($locale);
        
        return [
            'id' => $quiz->id,
            'title' => $translations['title'] ?? $quiz->title,
            'description' => $translations['description'] ?? $quiz->description,
            'questions' => $quiz->questions->map(function ($question) use ($locale) {
                $questionTranslations = $question->getI18nData($locale);
                return [
                    'id' => $question->id,
                    'type' => $question->type,
                    'question_text' => $questionTranslations['question_text'] ?? $question->question_text,
                    'question_data' => $question->question_data,
                    'answer_data' => $question->answer_data,
                    'points' => $question->score_points,
                    'is_required' => $question->is_required,
                ];
            }),
        ];
    }
}
