// Flash Messages JavaScript

document.addEventListener('DOMContentLoaded', function() {
    initializeFlashMessages();
});

document.addEventListener('livewire:init', function() {
    // Listen for Livewire events
    Livewire.on('autoRemoveMessage', function(messageId, duration) {
        setTimeout(() => {
            removeFlashMessage(messageId);
        }, duration);
    });
    
    Livewire.on('clearFlashMessages', function() {
        clearAllFlashMessages();
    });
});

function initializeFlashMessages() {
    // Initialize progress bars for existing messages
    initializeProgressBars();
    
    // Add click handlers for close buttons
    document.addEventListener('click', function(e) {
        if (e.target.closest('.flash-message-close')) {
            e.preventDefault();
            const messageElement = e.target.closest('.flash-message');
            const messageId = messageElement.id.replace('flash-', '');
            removeFlashMessage(messageId);
        }
    });
    
    // Auto-remove messages based on duration
    const messages = document.querySelectorAll('.flash-message');
    messages.forEach(message => {
        const duration = message.dataset.duration;
        if (duration && duration > 0) {
            setTimeout(() => {
                const messageId = message.id.replace('flash-', '');
                removeFlashMessage(messageId);
            }, parseInt(duration));
        }
    });
}

function initializeProgressBars() {
    const progressBars = document.querySelectorAll('.flash-message-progress');
    progressBars.forEach(progressBar => {
        const duration = parseInt(progressBar.dataset.duration);
        if (duration > 0) {
            progressBar.style.animationDuration = duration + 'ms';
        }
    });
}

function removeFlashMessage(messageId) {
    const messageElement = document.getElementById('flash-' + messageId);
    if (messageElement) {
        messageElement.classList.add('removing');
        setTimeout(() => {
            if (messageElement.parentNode) {
                messageElement.parentNode.removeChild(messageElement);
            }
        }, 300);
    }
}

function clearAllFlashMessages() {
    const messages = document.querySelectorAll('.flash-message');
    messages.forEach(message => {
        message.classList.add('removing');
        setTimeout(() => {
            if (message.parentNode) {
                message.parentNode.removeChild(message);
            }
        }, 300);
    });
}

// Utility functions for creating flash messages
window.FlashMessage = {
    success: function(message, title = null, duration = 5000) {
        if (window.Livewire) {
            window.Livewire.dispatch('flashMessage', 'success', message, title, duration);
        }
    },
    
    error: function(message, title = null, duration = 8000) {
        if (window.Livewire) {
            window.Livewire.dispatch('flashMessage', 'error', message, title, duration);
        }
    },
    
    warning: function(message, title = null, duration = 6000) {
        if (window.Livewire) {
            window.Livewire.dispatch('flashMessage', 'warning', message, title, duration);
        }
    },
    
    info: function(message, title = null, duration = 5000) {
        if (window.Livewire) {
            window.Livewire.dispatch('flashMessage', 'info', message, title, duration);
        }
    },
    
    clear: function() {
        if (window.Livewire) {
            window.Livewire.dispatch('clearMessages');
        }
    }
};

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Press 'Escape' to clear all flash messages
    if (e.key === 'Escape') {
        const flashContainer = document.querySelector('.flash-message-container');
        if (flashContainer && flashContainer.children.length > 0) {
            FlashMessage.clear();
        }
    }
});

// Auto-hide messages on page navigation
window.addEventListener('beforeunload', function() {
    clearAllFlashMessages();
});

// Handle page visibility changes
document.addEventListener('visibilitychange', function() {
    if (document.hidden) {
        // Pause animations when page is hidden
        const progressBars = document.querySelectorAll('.flash-message-progress');
        progressBars.forEach(progressBar => {
            progressBar.style.animationPlayState = 'paused';
        });
    } else {
        // Resume animations when page is visible
        const progressBars = document.querySelectorAll('.flash-message-progress');
        progressBars.forEach(progressBar => {
            progressBar.style.animationPlayState = 'running';
        });
    }
});

// Export functions for global access
window.removeFlashMessage = removeFlashMessage;
window.clearAllFlashMessages = clearAllFlashMessages;
